package de.ugoe.cs.eventbench.markov;

import java.util.ArrayList;
import java.util.List;

import de.ugoe.cs.eventbench.data.Event;


/**
 * This class implements a simple first-order markov state. 
 * Transitions are unlabled.
 * @author Steffen Herbold
 */
public class SimpleState extends State implements DotPrinter {
	
	List<Double> transitionProbs;
	List<State> toStates;
	
	// members for learning
	private int transitionsObserved;
	
	private String idShort = null;
	
	public SimpleState(String id, Event<?> action) {
		super(id, action);
		transitionsObserved = 0;
		toStates = new ArrayList<State>();
		transitionProbs = new ArrayList<Double>();
	}
	
	public SimpleState(String id, Event<?> action, String idShort) {
		this(id, action);
		this.idShort = idShort;
	}
	
	@Override
	public State getNextState() {
		double randVal = rand.nextDouble();
		double probSum = 0;
		int index = 0;
		while( index<transitionProbs.size() && probSum+transitionProbs.get(index) < randVal ) {
			probSum += transitionProbs.get(index);
			index++;
		}
		return toStates.get(index);
	}
	
	public void incTransTo(State state) {
		int index = toStates.indexOf(state);
		if( index==-1 ) {
			toStates.add(state);
			transitionProbs.add(0.0);
			index = toStates.size()-1;
		}
		// update trans probs
		for( int i=0 ; i<toStates.size() ; i++ ) {
			double currentProb = transitionProbs.get(i);
			double newProb = 0.0;
			if( i!=index ) {
				newProb = (currentProb*transitionsObserved)/(transitionsObserved+1);
			} else {
				newProb = ((currentProb*transitionsObserved)+1)/(transitionsObserved+1);
			}
			transitionProbs.set(i, newProb);
		}
		transitionsObserved++;
	}
	
	// get the transition probability to the given state
	public double getProb(State state) {
		double prob = 0.0;
		int index = toStates.indexOf(state);
		if( index>=0 ) {
			prob = transitionProbs.get(index);
		}
		return prob;
	}
	
	public String getShortId() {
		String shortId;
		if( idShort!=null ) {
			shortId = idShort;
		} else {
			shortId = getId();
		}
		return shortId;
	}

	@Override
	public void printDot() {
		final String thisSaneId = getShortId().replace("\"", "\\\"").replaceAll("[\r\n]","");
		System.out.println(" " + hashCode() + " [label=\""+thisSaneId+"\"];");
		for(int i=0 ; i<toStates.size() ; i++ ) {
			System.out.print(" "+hashCode()+" -> " + toStates.get(i).hashCode() + " ");
			System.out.println("[label=\"" + transitionProbs.get(i) + "\"];");
		}
		
	}
	
}