package de.ugoe.cs.util.console;

import java.io.IOException;

/**
 * <p>
 * Implements a simple console observer that prints normal text to
 * <code>stdout</code>, errors to <code>stderr</code> and reads from
 * <code>stdin</code>.
 * </p>
 * 
 * @author Steffen Herbold
 */
public class TextConsole implements ConsoleObserver {

	/**
	 * <p>
	 * In the debug mode, trace messages will be printed.
	 * </p>
	 */
	private boolean debugMode = true;

	/**
	 * <p>
	 * Creates a new text console and automatically registers it as observer.
	 * </p>
	 */
	public TextConsole() {
		Console.getInstance().registerObserver(this);
	}

	/**
	 * <p>
	 * Prints messages to <code>stdout</code>.
	 * </p>
	 * 
	 * @see ConsoleObserver#updateText(java.lang.String)
	 */
	public void updateText(String newMessage) {
		System.out.print(newMessage);
	}

	/**
	 * <p>
	 * Prints messages to <code>stderr</code>.
	 * </p>
	 * 
	 * @see ConsoleObserver#errStream(String)
	 */
	@Override
	public void errStream(String errMessage) {
		System.err.print(errMessage);
	}

	/**
	 * <p>
	 * Prints the stackrace of an exception to <code>stderr</code>
	 * </p>
	 * 
	 * @see ConsoleObserver#printStacktrace(Exception)
	 */
	@Override
	public void printStacktrace(Exception e) {
		e.printStackTrace();
	}

	/**
	 * <p>
	 * Prints messages to <code>stdout</code>. These messages are only printed,
	 * if the console is run in debug mode.
	 * </p>
	 */
	@Override
	public void trace(String traceMessage) {
		if (debugMode) {
			System.out.print(traceMessage);
		}
	}

	/**
	 * <p>
	 * Starts a new TextConsole. If the text console is started, it can be used
	 * not only to print message, but also to execute commands by reading
	 * <code>stdin</code>.
	 * </p>
	 * 
	 * @param debugMode
	 *            true, if the application is to run in debug mode, i.e. trace
	 *            messages will be printed
	 */
	public void run(boolean debugMode) {
		this.debugMode = debugMode;
		CommandExecuter exec = CommandExecuter.getInstance();
		while (true) {
			System.out.print("> ");
			String command = getCommand().trim();
			if (!command.equals("")) {
				exec.exec(command);
			}
		}
	}

	/**
	 * <p>
	 * Reads a new command from <code>stdin</code>.
	 * </p>
	 * 
	 * @return a string with a command
	 */
	protected String getCommand() {
		byte[] buffer = new byte[1024];
		try {
			System.in.read(buffer);
		} catch (IOException e) {

		}
		return new String(buffer);
	}
}
