package de.ugoe.cs.eventbench.efg.commands;

import java.security.InvalidParameterException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;

import de.ugoe.cs.eventbench.data.Event;
import de.ugoe.cs.eventbench.data.GlobalDataContainer;
import de.ugoe.cs.eventbench.models.FirstOrderMarkovModel;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.Console;
import edu.umd.cs.guitar.model.GUITARConstants;
import edu.umd.cs.guitar.model.IO;
import edu.umd.cs.guitar.model.data.EFG;
import edu.umd.cs.guitar.model.data.EventGraphType;
import edu.umd.cs.guitar.model.data.EventType;

/**
 * <p>
 * Command to that loads an EFG and creates a first-order Markov model with the
 * same structure.
 * </p>
 * 
 * @author Steffen Herbold
 * @version 1.0
 */
public class CMDefgToMM implements Command {

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.util.console.Command#run(java.util.List)
	 */
	@Override
	public void run(List<Object> parameters) {
		String filename;
		String modelname;
		try {
			filename = (String) parameters.get(0);
			modelname = (String) parameters.get(1);
		} catch (Exception e) {
			throw new InvalidParameterException();
		}

		EFG efg = (EFG) IO.readObjFromFile(filename, EFG.class);

		// extracting the events from the EFG
		List<EventType> efgEvents = efg.getEvents().getEvent();
		List<Event<?>> myEvents = new ArrayList<Event<?>>(efgEvents.size());
		for (EventType event : efgEvents) {
			/*
			 * Steffen (Question): I think the widgetId is only a hash value
			 * identifying the target Is it sufficient as a target or should it
			 * be somehow resolved into something more meaningful? If so, how?
			 * Where is it available? There is some kind of GUI file with this
			 * information, right? Should information of the GUI file be
			 * resolved or the GUI file simply be associated with the EFG
			 * internally. The best solution for this probably depends on the
			 * test-case structure, which I have not analyzed yet.
			 */

			// both widget ID and eventType are always null in the sample EFGs
			// ...
			String eventTarget = event.getWidgetId();
			//String eventType = event.getType();

			/*
			 * What is the Action? What is the difference between the Action and
			 * the Type?
			 */
			String eventAction = event.getAction();

			Event<?> myEvent = new Event<Object>(eventAction);
			myEvent.setTarget(eventTarget);
			myEvents.add(myEvent);
		}

		// extracting graph structure from the EFG
		/*
		 * getEventGraph returns an adjacency matrix, i.e., a square matrix of
		 * efgEvents.size(), where a 1 in row i, column j means an edge
		 * efgEvents.get(i)->efgEvents.get(j) exists.
		 */
		EventGraphType efgGraph = efg.getEventGraph();

		FirstOrderMarkovModel model = new FirstOrderMarkovModel(new Random());
		Collection<List<Event<?>>> subsequences = new LinkedList<List<Event<?>>>();

		/*
		 * Code adapted from package
		 * edu.umd.cs.guitar.testcase.plugin.TCPlugin#parseFollowRelations()
		 * (part of testcase-generator-core)
		 */
		int efgSize = efgEvents.size();
		for (int row = 0; row < efgSize; row++) {
			for (int col = 0; col < efgSize; col++) {
				int relation = efgGraph.getRow().get(row).getE().get(col);

				// otherEvent is followed by currentEvent
				if (relation != GUITARConstants.NO_EDGE) {
					List<Event<?>> edge = new LinkedList<Event<?>>();
					edge.add(myEvents.get(row));
					edge.add(myEvents.get(col));
					subsequences.add(edge);

					/*
					 * Steffen (Question): What is the purpose of this if? What
					 * is the difference between a normal and a reaching edge?
					 * if (relation == GUITARConstants.REACHING_EDGE &&
					 * !otherEvent.getEventId().equals(
					 * currentEvent.getEventId())) { I probably don't need this
					 * anyways, since for usage analysis only successors are
					 * relevant Vector<EventType> p = null;//
					 * preds.get(otherEvent); if (p == null) { p = new
					 * Vector<EventType>(); }
					 * 
					 * p.add(currentEvent); preds.put(otherEvent, p); }
					 */
				}
			}
		}
		model.train(subsequences);
		GlobalDataContainer.getInstance().addData(modelname, model);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.util.console.Command#help()
	 */
	@Override
	public void help() {
		Console.println("Usage: efgToMM <filename> <modelname>");
	}

}
